/* recursive decent parser derived from <groan> Herbert Schildt's code */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#define DELIMITER   1
#define NUMVAR      2
#define NUMBER      3
#define COMMAND     4
#define STRING      5
#define QUOTE       6
#define STRVAR      7
#define NUMFUNC     8
#define STRFUNC     9

#define EOL         10
#define FINISHED    11

#define END         12

typedef struct commands {       /* keyword lookup table */
  char *command;
  int  tok;
} TABLE;

char           *parse_error = NULL;
long            variables[26] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0
};
char            strvariables[26][255] = {
  "","","","","","","","","","",
  "","","","","","","","","","",
  "","","","","",""
};
char           *prog;
char            token[80];
int             token_type;
int             tok;

TABLE ctable[] = {              /* must be asc. sorted! */
  "end", END,
  "", END                       /* end of table marker */
};

TABLE nftable[] = {             /* must be asc. sorted! */
  "", END                       /* end of table marker */
};

TABLE sftable[] = {             /* must be ascii sorted! */
  "", END                       /* end of table marker */
};

#define CTABLELEN   1           /* length of ctable[] */
#define NFTABLELEN  0           /* length of nftable[] */
#define SFTABLELEN  0           /* length of sftable[] */

void get_numexp         (long *result);
char * get_strexp       (void);
void num_level2         (long *result);
void num_level3         (long *result);
void num_level4         (long *result);
void num_level5         (long *result);
void num_level6         (long *result);
void primitive          (long *result);
void arith              (char o, long *r, long *h);
int  get_token          (void);
void putback            (void);
int  look_up            (char *s,TABLE *table,int tablelength);
int  isdelim            (char c);
int  iswhite            (char c);
long find_numvar        (char *s);
char * find_strvar      (char *s);
long numeric_function   (int tok);
char * string_function  (int tok);



/* Entry point to string parser */

char * get_strexp (void)
{

}

/* Entry point to long integer parser */

void get_numexp (long *result)
{
  long            value;

  parse_error = NULL;
  get_token ();
  if (parse_error)
    return;
  if (!*token) {
    parse_error = "no expression present";
    return;
  }
  num_level2 (result);
  if (!parse_error)
    putback ();
}

/* Add/subtract/bit manipulate two terms */

void num_level2 (long *result)
{
  register char   op;
  long            hold;

  num_level3 (result);
  if (parse_error)
    return;
  while ((op = *token) == '+' || op == '-' || op == '|' ||
         op == '&' || op == '!') {
    get_token ();
    if (parse_error)
      return;
    num_level3 (&hold);
    if (parse_error)
      return;
    arith (op, result, &hold);
  }
}

/* Multiply or divide two factors */

void num_level3 (long *result)
{
  register char   op;
  long            hold;

  num_level4 (result);
  if (parse_error)
    return;
  while ((op = *token) == '*' || op == '/' || op == '%') {
    get_token ();
    if (parse_error)
      return;
    num_level4 (&hold);
    if (parse_error)
      return;
    arith (op, result, &hold);
  }
}

/* Process long integer exponent */

void num_level4 (long *result)
{
  long            hold;

  num_level5 (result);
  if (parse_error)
    return;
  if (*token == '^') {
    get_token ();
    if (parse_error)
      return;
    num_level4 (&hold);
    if (parse_error)
      return;
    arith ('^', result, &hold);
  }
}

/* Is a unary + or - or ~ (not) */

void num_level5 (long *result)
{
  register char   op;

  op = 0;
  if ((token_type == DELIMITER) && *token == '+' || *token == '-' ||
       *token == '~') {
    op = *token;
    get_token ();
    if (parse_error)
      return;
  }
  num_level6 (result);
  if (parse_error)
    return;
  if (op == '-')
    *result = -(*result);
  else if (op == '~')
    *result = ~(*result);
}

/* Process parenthesized expression */

void num_level6 (long *result)
{
  if ((*token == '(') && (token_type == DELIMITER)) {
    get_token ();
    if (parse_error)
      return;
    num_level2 (result);
    if (parse_error)
      return;
    if (*token != ')') {
      parse_error = "unbalanced parentheses";
      return;
    }
    get_token ();
    if (parse_error)
      return;
  }
  else
    primitive (result);
}

/* Find value of number or variable */

void primitive (long *result)
{
  int             numvar;
  long            value;

  switch (token_type) {
    case NUMVAR:
      *result = find_numvar (token);
      if (parse_error)
        return;
      get_token ();
      return;

    case STRVAR:
      *result = atol(find_strvar(token));
      if(parse_error)
        return;
      get_token();
      return;

    case QUOTE:
      *result = atol(token);
      get_token();
      return;

    case NUMBER:
      *result = atol (token);
      get_token ();
      return;

    case NUMFUNC:
      *result = numeric_function(tok);
      return;

    case STRFUNC:
      *result = atol(string_function(tok));
      return;

    default:
      parse_error = "syntax error";
      return;
  }
}


/* Perform the specified arithmetic */

void arith (char o, long *r, long *h)
{
  long            t,
                  ex;

  switch (o) {
    case '-':
      *r -= *h;
      break;
    case '+':
      *r += *h;
      break;
    case '*':
      *r *= (*h);
      break;
    case '/':
      *r /= *h;
      break;
    case '%':
      t = (*r) / (*h);
      *r -= (t * (*h));
      break;
    case '^':
      ex = *r;
      if (*h == 0) {
        *r = 1;
        break;
      }
      for (t = *h - 1; t > 0; --t)
        *r = (*r) * ex;
      break;
    case '&':
      *r &= *h;
      break;
    case '|':
      *r |= *h;
      break;
    case '!':                   /* xor */
      *r ^= *h;
      break;
    case '{':                   /* shift left */
      *r <<= *h;
      break;
    case '}':                   /* shift right */
      *r >>= *h;
      break;
    default:
      parse_error = "invalid operator";
  }
}

/* Get a token */

int get_token (void)
{
  register char  *temp;

  token_type = 0;
  tok = 0;
  temp = token;

  if (*prog == '\0') {          /* End of input */
    *token = 0;
    tok = FINISHED;
    return (token_type = DELIMITER);
  }

  while (iswhite (*prog))
    ++prog;                     /* skip over white space */

  if (*prog == '\r') {          /* crlf */
    prog += 2;
    tok = EOL;
    *token = '\r';
    token[1] = '\n';
    token[2] = 0;
    return (token_type = DELIMITER);
  }

  if (strchr ("+-*^/=(),><!|&%;{}", *prog)) {  /* delimiter */
    *temp = *prog;
    prog++;                     /* advance to next position */
    temp++;
    *temp = 0;
    return (token_type = DELIMITER);
  }

  if (*prog == '\"') {          /* Quoted string */
    prog++;
    while (*prog != '\"' && *prog != '\r')
      *temp++ = *prog++;
    if (*prog == '\r') {
      parse_error = "unbalanced parentheses";
      return 0;
    }
    prog++;
    *temp = '\0';
    return (token_type = QUOTE);
  }

  if (isdigit (*prog)) {        /* Number */
    while (!isdelim (*prog))
      *temp++ = *prog++;
    *temp = '\0';
    return (token_type = NUMBER);
  }

  if (isalpha (*prog)) {
    while (!isdelim (*prog))
      *temp++ = *prog++;
    token_type = STRING;
  }

  *temp = '\0';

  /* See if a string is a command, function or variable */

  if (token_type == STRING) {
    if(*token == '$')
      token_type = STRVAR;
    else if(*token == '#')
      token_type = NUMVAR;
    else if(*token == ':') {
      tok = look_up (token + 1,ctable,CTABLELEN);
      if (tok)
        token_type = COMMAND;     /* is a command */
    }
    else if(*token == '@') {
      if(*(token + 1) == '#') {
        tok = look_up(token + 2,nftable,NFTABLELEN);
        if(tok)
          token_type = NUMFUNC;   /* is a numeric function */
      }
      else if(*(token + 1) == '$') {
        tok = look_up(token + 2,sftable,SFTABLELEN);
        if(tok)
          token_type = STRFUNC;   /* is a string function */
      }
    }
  }

  return token_type;
}


/* Return a token to input stream */

void putback (void)
{
  char           *t;

  t = token;
  for (; *t; t++)
    prog--;
}


/* Look up a token's internal representation in a token table */

int look_up (char *s,TABLE *table,int tablelength)
{
    int l, u;       /* lower and upper limits of binary search */
    int i;          /* current search index */
    int cresult;    /* result of comparison */

    /* set current search limit as entire list */

    l = 0;
    u = tablelength - 1;

    /* get the midpoint! */

    while (u >= l) {
        i = (l + u) >> 1;

        /* do the comparison */
        cresult = stricmp(s, table[i].command);
        if (cresult == 0)
            return ctable[i].tok;
        if (cresult < 0)
            u = i - 1;
        else
            l = i + 1;
    }
    return 0;
}

/* Return true if c is a delimiter */

int isdelim (char c)
{
  if (strchr (" ;,+-<>/*%^=()!&|\t\r", c) || !c)
    return 1;
  return 0;
}

/* Return 1 if c is space or tab */

int iswhite (char c)
{
  if (c == ' ' || c == '\t')
    return 1;
  else
    return 0;
}

/* find the value of a numeric variable */

long find_numvar (char *s)
{
  if (!isalpha (*(s + 1))) {
    parse_error = "not a variable";
    return 0;
  }
  return variables[toupper (*(s + 1)) - 'A'];
}

/* find the contents of a string variable */

char *find_strvar (char *s)
{
  if(!isalpha(*(s + 1))) {
    parse_error = "not a variable";
    return "";
  }
  return strvariables[toupper(*(s + 1)) - 'A'];
}

/* return value of numeric function */

long numeric_function (int tok)
{

}

/* return contents of string function */

char * string_function (int tok)
{

}
